package RePEc::Index::History::Handle;

#  the logic of tracking handle's history and status

use strict;
use warnings;

use Carp::Assert;

use Events;

Events -> register_event ( 'RePEc::Index::RECORD::PROCESS' );
Events -> register_event ( 'RePEc::Index::RECORD::DELETE' );

Events -> register_event_handler( 'RePEc::Index::Update::RECORD',            
                                  \&event_record             );
Events -> register_event_handler( 'RePEc::Index::Update::RECORD_DISAPPEAR',  
                                  \&event_record_disappear   );


use RePEc::Index::Storage qw(
                             &load_record_from_db_txn
                             &load_record_from_db_txn_readonly
                             &save_record_to_db_txn
                             &delete_record_from_db_txn
                             &start_transaction
                             &commit_transaction
                            );



Events -> register_event( 'RePEc::Index::RECORD::NEW', 
                          "a good record came in, with no conflict" );

Events -> register_event( 'RePEc::Index::RECORD::OLD', 
                          "a good record came in, with no conflict" );

Events -> register_event( 'RePEc::Index::RECORD::CHANGED', 
                          "a good record came in, with no conflict" );

Events -> register_event( 'RePEc::Index::RECORD::CONFLICT', 
                          "a record present event resulted in a conflict" );

Events -> register_event( 'RePEc::Index::RECORD::CLEAR', 
                          "a record lost event resulted in a conflict clearance" );


sub clear_undefined ($) {
  my $ar = shift;  my $i = 0;
  while ( $i < scalar @$ar ) {
    my $v = $ar ->[$i];
    if ( not defined $v ) { splice @$ar, $i, 1;
                            next; }
    $i ++;
  }
}

use vars qw( $ID $RECORD $UPDATE $CLEARED $TYPE $REC_HISTORY $UTIME );

### a debugging tool
sub p { no warnings; print @_, "\n"; }

sub new_rec_history {
  my $class  = shift;
  my $handle = shift;
  my $data   = shift;
  
  my $self  ;

  if ( defined $data ) {
    assert( $data->{handle} eq $handle );
    $self = { %$data };

  } else {
    $self = {
             present => [],
             last_changed => undef,
             last_processed  => undef,
             handle  => $handle,
#            type    => $type,
             history => [],
             };
  }

  bless $self, $class;

  return $self;
}
### the end #################################################################




# generated at RePEc::Index::Update
sub event_record {
  shift;
  my $id = shift;
  my $record = shift;
  my $type   = shift;
  my $filename = shift;
  my $pos      = shift;
  my $checksum = shift;
  my $update   = shift;

  $ID     = $id ;
  $TYPE   = $type;
  $RECORD = $record;
  $UPDATE = $update;
  $UTIME  = $UPDATE ->{SESSION};

  my $hdb = $update->{history_db};
  my $txn = start_transaction();

  $REC_HISTORY = my $hrecord = load_record_from_db_txn( $txn, $hdb, $id );

  if ( not $hrecord ) {
    $hrecord = __PACKAGE__ -> new_rec_history( $id );
  }

  $hrecord -> update_session_time( $UTIME );

  my @fight_back = event_rec_present( $hrecord, $filename, $pos, $checksum, $type );

  if ( scalar @fight_back ) {
    $update -> add_to_queue( @fight_back );
    $update -> requeue( $filename );
  }

  save_record_to_db_txn( $txn, $hdb, $id, $hrecord );

  my $r = commit_transaction( $txn );

  return $r;
}



#############################################################################
##  s u b   E V E N T   R E C   P R E S E N T 
#############################################################################
##  (a) event
sub event_rec_present {
  my $self = shift;
  my $file = shift;
  my $position = shift;
  my $checksum = shift;
  my $type     = shift;
  
  my $id   = $self->{handle};

#  $self -> dump();

  assert( $file and defined( $position ) and $checksum );

  my $was_conflict = $self->{conflict};
#  p "rec $self->{handle}, file $file, conflict: ", $self->{conflict}; 

  my $session_time = $self->{session_time};
  my $history = $self->{history};

  push @$history, [ $session_time, "present", $file, $position, $checksum ];

  # now check if there is a conflict
  my $present = $self -> {present};
  my $files_to_check = {};

  if ( defined $self->{files_to_check} ) {
    my $ftc = $files_to_check = $self->{files_to_check} ;
    if ( exists $ftc ->{$file} ) {
      delete $ftc ->{$file};
    }
  }

  my $old_checksum;

  if ( not $was_conflict ) {
 
    if ( defined $present 
         and scalar @$present ) {
      $old_checksum = $present->[0]->[2];
    }
  }

  # need to check $self->{present} array 

  # if there are any "new" items, ie, this session's, then it's a conflict. 

  my $append_record_to_present = 1;
  my $conflict    = [];
  my $time_status = 'new';
  my @lost_present_file;

  my @current; ### list of certainly-valid present items
  push @current, [ $file, $position, $checksum ];

  my $replaced;


  foreach ( @$present ) {
    my $fn = $_->[0];
    my $po = $_->[1];
    my $t  = $_->[3];  ### last seen in that file at time t

    if ( $t > $session_time ) {
      ### to avoid creating hard-to-deal-with nonsense data, abort
      print "record was seen in the future: $t\naborting";
      return ();
    }

    if ( $t < $session_time ) {  ### record seen in the past
      $time_status = "old"; 
    }      


    my $replace_this_item;

    if ( $fn eq $file ) {
      ### 
      $replace_this_item = 1;

      if ( $t == $session_time 
           and $po < $position ) {
        ### a previous mention of the same id in the same file,
        ### certain sign of a conflict
        push @current, $_;
        undef $replace_this_item;
      }

    } else {
      ### some other file 

      p "file: $fn";

      # Is this item still valid?  Does that file still contain the record?
      # If we knew it had in the past, and it didn't change since, then it is
      # valid.

      # So, has this file been processed recently?  (Recently means after last
      # modification, as known to the filesystem.)
      # If not, queue it for processing...

      my $status = $UPDATE -> file_needs_update( $fn );

      if ( $status ) { # will queue
        p "will check it";
        push @current, $_;
        $files_to_check -> {$fn} = 1;
        
      } else {
        p "no need to check it to draw conclusions";

        # Is that file known to contain a record with the same id?
        # CONSISTENCY CHECK
        
        my $frec = $UPDATE -> get_file_record( $fn );
        if ( $frec ) {
          my $present_there;
          my $list = $frec -> templates_list;
          foreach ( @$list ) {
            if ( $_ eq $id ) { $present_there = 1; }
          }
          if ( $present_there ) {
            p "the record is still there";
            push @current, $_;

          } else {
            p "the record is no longer there";
            push @lost_present_file, $fn;
          }
        } else {
          p "the file is no longer there";
          push @lost_present_file, $fn;
        }

      }
    }

    if ( not $replaced 
         and $replace_this_item ) {
      @$_ = ( $file, $position, $checksum, $session_time );
      $append_record_to_present = 0;
      $replaced = 1;
    }

  }

  if ( $append_record_to_present ) {
    push @$present, [ $file, $position, $checksum, $session_time ];
  }

  if ( scalar @lost_present_file ) {
    $self->{files_to_check} = $files_to_check;
    foreach ( @lost_present_file ) {
      $self -> lost_template( $_ );
    }
    $files_to_check = $self->{files_to_check};
  }


  my @event_data = ( $id, $self );

  if ( not $was_conflict 
       and scalar( @current ) > 1 ) {
    found_id_conflict( $self );

  } elsif ( not $self->{conflict} ) {

    $self->{type} = $type;  # remember the record type
    my $changed;

    if ( defined $old_checksum 
         and $old_checksum ne $checksum ) { $changed = 1; }

    if ( exists $self->{cleared} ) {
      $changed = 1; 
      delete $self->{cleared};
    }

    if ( $changed ) {
      found_rec_change( $self, $file, $position, $checksum );

    } else {
      if ( $time_status eq 'old' ) {
        Events->RePEc::Index::RECORD::OLD ( @event_data );

      } elsif ( $time_status eq 'new' ) {
        Events->RePEc::Index::RECORD::NEW ( @event_data );
      }
    }
  }

  if ( defined $files_to_check 
       and scalar keys %$files_to_check ) {

    my $ftc = $self->{files_to_check} = $files_to_check;
    if ( exists $ftc ->{$file} ) {
      delete $ftc ->{$file};
    }
    return keys %$files_to_check; 
  } 

  delete $self->{files_to_check};
  return;
}
### the end #################################################################






#############################################################################
##  s u b   C H A N G E    E V E N T 
#############################################################################

sub found_rec_change {
  my $self = shift;

  my $session_time = $self->{session_time};
  my $history = $self->{history};

  # log it to the history
  push @$history, [ $session_time, "change" ];  # ??? XXX incomplete

  $self->{last_changed} = $session_time;
  Events->RePEc::Index::RECORD::CHANGED ( $ID, $self );

}
### the end #################################################################




#############################################################################
##  s u b    F O U N D   I D   C O N F L I C T
#############################################################################
sub found_id_conflict {
  my $self    = shift;
  my $present = $self->{present};

  assert( scalar( @$present ) > 1 ) if DEBUG;

  my $session_time = $self->{session_time};
  my $history = $self->{history};

  delete $self->{type};
  delete $self->{last_processed};
  
  my @conflicting_files ;

  my $conflict_is_well_known = 0;

  my $conflict = $self->{conflict};
  if ( not defined $conflict ) { $conflict = {}; }
  else { $conflict_is_well_known = 1; }


  # put data into the conflict hash
  for ( @$present ) {
    my $t = $_->[3];
    next if $t < $session_time ;

    my $f = $_->[0];
    my $p = $_->[1];
    my $c = $_->[2];

    # conflict 
    if ( $conflict_is_well_known ) {
      if ( not exists $conflict ->{ $f } 
           or not exists $conflict ->{ $f } -> { $p }
        ) { 
        $conflict_is_well_known = 0;
      }
    }
    $conflict -> { $f } -> { $p } = [ $c, $session_time ] ;
  }

  $self->{conflict} = $conflict;

  return if $conflict_is_well_known;

  # build a history record data
  foreach my $filename ( keys %$conflict ) {
    foreach my $pos ( keys %{ $conflict->{$filename} } ) {
      my $cs    = $conflict->{$filename} {$pos} [0];
#      my $stime = $conflict->{$filename} {$pos} [1];
      push @conflicting_files, [ $filename, $pos, $cs ] ;
    }
  }

  # log it to the history
  push @$history, [ $session_time, "conflict", @conflicting_files ];



  ### save the conflict in the conflict_db
  my $one = 1;

  my $txn = start_transaction();
  save_record_to_db_txn( $txn, $UPDATE->{conflict_db}, $ID, \$one );
  commit_transaction( $txn );

  Events->RePEc::Index::RECORD::CONFLICT ( $ID, $self );

  clear_record( $self, $ID );
}
### the end #################################################################


#############################################################################
##  s u b   I D   C O N F L I C T   C L E A R E D
#############################################################################
sub id_conflict_cleared {
  my $self = shift;

  if ( exists $self->{conflict} ) {
    delete $self ->{conflict};
  }

  my $session_time = $self ->{session_time};
  my $history      = $self ->{history};
  my $present      = $self ->{present};

  # log it to the history
  push @$history, [ $session_time, "clear" ]; 
  $self ->{cleared} = 1;


  # deleting the conflict record

  my $txn = start_transaction();
  delete_record_from_db_txn( $txn, $UPDATE->{conflict_db}, $ID );
  commit_transaction( $txn );

  Events-> RePEc::Index::RECORD::CLEAR( $self->{handle}, $self );
  
  assert( scalar( @$present ) == 1 );
  
  $UPDATE -> add_to_queue( $present->[0][0] );
}

### the end #################################################################




Events->register_event_handler( "RePEc::Index::RECORD::NEW", 
                                \&save_and_process_record );
Events->register_event_handler( "RePEc::Index::RECORD::CHANGED",  
                                \&save_and_process_record );
Events->register_event_handler( "RePEc::Index::RECORD::OLD", 
                                \&event_record_old );

# generated at RePEc::Index::History::Handle


sub save_and_process_record {
  shift;

  ### storing the record itself in the record_db
  my ( $id, $hrecord ) = @_;

  if ( $UPDATE->{keep_records_db} ) {
    my $txn = start_transaction();
    save_record_to_db_txn( $txn, $UPDATE->{records_db}, $ID, [ $RECORD, $TYPE ] );
    commit_transaction( $txn );
  }

  if ( $UPDATE->{proc} ) {
    foreach my $proc ( @{ $UPDATE->{proc} } ) {
      $proc -> process_record( $ID, $TYPE, $RECORD );
    }
    Events-> RePEc::Index::RECORD::PROCESS( $ID, $TYPE, $RECORD );
    $hrecord -> processed();
  }
}


sub clear_record {
  my $self = shift;
  my $id   = shift;

  if ( $UPDATE->{proc} ) {
    foreach my $proc ( @{ $UPDATE->{proc} } ) {
      $proc -> delete_record( $id );
    }
  }
  Events-> RePEc::Index::RECORD::DELETE( $id, $self );
}


sub event_record_old {
  shift;
  my $id = shift;
  my $history = shift;
  
  assert( $history );
  my $last_changed   = $history->{last_changed} 
    || $history->{history}->[0]->[0];
  my $last_processed = $history->{last_processed};

  my $too_old_treshold = $UPDATE->{TOO_OLD_IS};
  my $session          = $UPDATE->{SESSION};

  assert( defined $too_old_treshold );
  assert( $session );

  my $old_files = $session - $too_old_treshold;


  if ( $UPDATE->{proc} ) {

    if ( not defined $last_processed
         or $last_processed < $last_changed 
         or $last_processed < $old_files ) { 

      save_and_process_record( 0, $id, $history );
    }
  }
}





# generated at RePEc::Index::Update

sub event_record_disappear {
  shift;
  my $id = shift;
  my $filename = shift;
  my $update   = shift;

  $ID = $id;
  $UPDATE = $update;

  assert( $UPDATE );
  assert( $ID );

  my $db = $update->{history_db};

  my $txn = $UPDATE -> {CURRENT_TXN};
  my $transaction_responsible = 0;
  if ( not $txn ) {
    $txn = start_transaction();
    $transaction_responsible = 1;
  }


  my $hrecord = load_record_from_db_txn( $txn, $db, $id );

  if ( not $hrecord ) {
    warn "can't load $id history record from db";
    return 0;
    $hrecord = __PACKAGE__ -> new_rec_history( $id );
  } 

  $hrecord -> update_session_time( $update->{SESSION} );

  $hrecord -> lost_template( $filename );
  my $present = $hrecord->{present};

  if ( not scalar @$present ) {
    # the template lost completely

    if ( $UPDATE->{keep_records_db} ) {
      ### deleting the record from the records_db
      my $recdb = $update->{records_db};
      delete_record_from_db_txn( $txn, $recdb, $id );
    }
    clear_record( $hrecord, $id );
  }

  save_record_to_db_txn( $txn, $db, $id, $hrecord );

  my $ret = 1;
  if ( $transaction_responsible ) {
    $ret = commit_transaction( $txn );
  }
  return $ret;
}





#############################################################################
##  s u b   U P D A T E    S E S S I O N    T I M E 
#############################################################################
sub update_session_time {
  my $self = shift;
  my $time = shift;
  assert( $time );
  $self -> {session_time} = $time;
}
### the end #################################################################


sub dump {
  my $self = shift;
  my $id   = $self->{handle};
  my $history  = $self->{history};
  my $present  = $self->{present};
  my $conflict = $self->{conflict};
  my $ftc      = $self->{files_to_check};
  my $type     = $self->{type};
  my $last_proc = $self->{last_processed};
  my $last_chan = $self->{last_changed};

  # ... 
  
  p "RECORD $id:";
  p "\tpresent:";
  foreach ( @$present ) {
    my ( $f, $p, $c, $s ) = @$_;
    p "\t\tfile $f, pos $p ($s)";
  }
  p "\tconflict:";
  foreach ( keys %$conflict ) {
    my ( $f, $p ) = ( $_, $conflict->{$_} );
    p "\t\tfile $f, pos ", join( ' ', keys %$p );
  }
  p "---";
}




#############################################################################
##  s u b   P R O C E S S E D     E V E N T 
#############################################################################
sub processed {
  my $self = shift;

  my $session_time = $self->{session_time};
  my $history = $self->{history};

  # log it to the history
  push @$history, [ $session_time, "processed" ];  

  $self->{last_processed} = $session_time;

}
### the end #################################################################


#############################################################################
##  s u b   L O S T    T E M P L A T E 
#############################################################################
sub lost_template {
  my $self = shift;
  my ( $file ) = @_;

  my $position;  # we need to find that out

  my $session_time = $self ->{session_time};
  my $history      = $self ->{history};
  my $id           = $self ->{handle};

  assert( $file );

  # check and update files_to_check list

  if ( defined $self->{files_to_check} ) {

    foreach ( keys %{ $self->{files_to_check} } ) {
      if ( $_ eq $file ) {
        delete $self->{files_to_check}->{$file};
      }
    }
  }


  my $present = $self -> {present};

  ###  we need to examine $self ->{present} to find the lost
  ###  template's data: position and checksum.  That should be pretty
  ###  simple as in $self->{present} we have all needed information.
  ###  We only need to extract it.  


  assert( $session_time );

  my $fileknown;
  foreach ( @$present ) {
    my ( $f, $p, $c, $stime ) = @$_;
    if ( $f eq $file ) {
      $fileknown = 1;
      if ( $stime < $session_time ) {
        undef $_;
        $position = $p;
        last;
      } else {
      }
    }
  }

  clear_undefined $present;

  if ( not $fileknown ) {
    warn "wrong lost_template() event: record $id didn't know about file $file";
    return 0;
  }


  # log it to the history
  push @$history, [ $session_time, "lost template", $file, $position ]; 
  # XXX incomplete: checksum should also be here?
  
  
  if ( defined $self->{conflict} ) {
    if ( scalar( @$present ) <= 1 ) {
      id_conflict_cleared( $self );

    } else {
      if ( ref ( $self->{conflict} ) eq 'HASH'
           and exists $self->{conflict} {$file}
          ) {
        # if there is a conflict record for this file...
        delete $self->{conflict} {$file} { $position } ; # delete old position's item
      } 
    }
  }


}
### the end - lost template #################################################


#############################################################################
##  s u b   L O S T    F I L E 
#############################################################################
sub lost_file {
  my $self = shift;

  my ( $file ) = @_;

  my $session_time = $self->{session_time};
  my $history = $self->{history};

  # log it to the history
  push @$history, [ $session_time, "lost file", $file ]; 

  # check and update files_to_check list
  if ( defined $self->{files_to_check} ) {
    foreach ( keys %{ $self->{files_to_check} } ) {
      if ( $_ eq $file ) {
        delete $self->{files_to_check}->{$file};
      }
    }
  }

  my $present = $self -> {present};
  
  my $found;  
  foreach ( @$present ) {
    my ( $f, $p ) = @$_;
    if ( $f eq $file ) { undef $_; $found = 1; }
  }
  
  clear_undefined $present;
  if ( not $found ) { warn "wrong lost_file() event"; }

  if ( defined $self->{conflict} ) {
    if ( scalar( @$present ) <= 1 ) {
      id_conflict_cleared( $self );

    } else {
      if ( ref ( $self->{conflict} ) eq 'HASH'
           and exists $self->{conflict} {$file}
         ) {
        # if there is a conflict record for this file...
        delete $self->{conflict} {$file} ; # delete 
      } 
    }
  }


}
### the end #################################################################





## accessor methods

#############################################################################
##  s u b   H A N D L E 
#############################################################################
sub handle {
  my $self = shift;
  return $self->{handle};
}
### the end #################################################################

#############################################################################
##  s u b   C O N F L I C T 
#############################################################################
sub conflict {
  my $self = shift;
  return $self->{conflict};
}
### the end #################################################################


#############################################################################
##  s u b   P R E S E N T
#############################################################################
sub status_present {
  my $self = shift;
  return $self->{present};
}
### the end #################################################################

#############################################################################
##  s u b   H I S T O R Y
#############################################################################
sub history {
  my $self = shift;
  return $self->{history};
}
### the end #################################################################

#############################################################################
##  s u b   S T A T U S 
#############################################################################
sub status {
  my $self = shift;
  return $self->{status};
}
### the end #################################################################

#############################################################################
##  s u b   L A S T    C H A N G E D
#############################################################################
sub last_changed {
  my $self = shift;
  return $self->{last_changed};
}
### the end #################################################################


#############################################################################
##  s u b   L A S T    P R O C E S S E D
#############################################################################
sub last_processed {
  my $self = shift;
  return $self->{last_processed};
}
### the end #################################################################


1;

__END__

#############################################################################
##  s u b   
#############################################################################
sub  {
}
### the end #################################################################


