package RePEc::Index::UpdateClient;

use Exporter;
use base qw( Exporter );

###  interface

@EXPORT_OK = qw( &send_update_request );

###  implementation

use strict;
use warnings;
use Carp::Assert;

use vars qw( $socket $home );

use Socket;

if ( $RePEc::Index::Socket ) { 
  $socket =  $RePEc::Index::Socket;

} else {
  eval q! require RePEc::Index::local_setup; !;
  $socket = $RePEc::Index::local_setup::ri_socket;
}

use RePEc::Index::Config;

$home = $RePEc::Index::Config::HOMEDIR;

sub set_server_socket { 
  my $s = shift;
  $socket = $s;
}


sub send_update_request {

  my $collection = shift || die;
  my $file       = shift || die;
  my $force      = shift || 0;
  
  assert( $socket );

  my $forked;
  my $attempt = 0;
  my $atlimit = 20; 
  my $wait    = 45; 

  my $success;
  
  my $pid = $$;

  while ( 1 ) {

    eval { 
      socket ( Server, PF_UNIX, SOCK_STREAM, 0 )
        or die "unable to create socket: $!";
      
      connect ( Server, sockaddr_un ( $socket ) )
        or die "unable to connect: $! ($socket)";
      
      $| = 1;
      
      print Server "SOURCE $0 [$pid]\nCOLLECTION $collection\nUPDATE $file\n",
        ($force) ? "FORCE $force\n" : "",
          "\n";
      
      close Server;
    };

    $attempt ++;
    
    if ( not $@ ) {
      $success = 1;
      last;
    }

    if ( $attempt >= $atlimit ) { last; }

    if ( $@ 
         and $@ =~ /^unable to connect: / 
         and not $forked ) {

      my $force = $force ? " (forced)" : '';
      ## do not warn if running on the web, this would 500
      if ($ACIS::FCGIReq) {
        ## fixme: report the failure to connect to the maintainer
        return;
      }
      warn "[$$] Forking a request: $collection: $file$force\n";
         
      my $f = fork;
      
      if ( $f ) {
        wait;
        return 0;

      } elsif ( $f == 0 ) {
        
        $forked = 1;
        $pid = "$pid/$$";
        require Proc::Daemon;
        Proc::Daemon::Init();
        $pid = "$pid/$$";
        
      } else {
        warn "Can't fork!";
      }
    }

    sleep $wait;
  }

  if ( not $success ) {
    if ( open ERRLOG, ">>", "$home/client_failed.log" ) {
      print ERRLOG scalar localtime, " Failed to connect to server:\n";
      print ERRLOG "SOURCE $0 [$pid]\nCOLLECTION $collection\nUPDATE $file\n",
        ($force) ? "FORCE $force\n" : "", "\n";
      close ERRLOG;
    }
  }

  if ( $forked ) { exit; }
  return 1;
}

1;
