#!/usr/bin/perl

# Usage:
#
#   perl release.pl [VERSION]
# 
# This should update Makefile.PL 


use strict;

sub get_cur_version;
sub increment_version;
sub get_new_version;
sub save_version_in_makefile_pl;
sub submit_makefile_pl;
sub make_release;


sub get_cur_version {
  my $file = `cat Makefile.PL`;
  $file =~ m/VERSION\s*=>\s*['"](\d+[\.\d+]+)['"]/;
  if ($1) {
    return $1;
  }
}

sub get_module_name {
  my $file = `cat Makefile.PL`;
  $file =~ m/\sNAME\s*=>\s*['"]([^\'\"]+)['"]/;
  if ($1) {
    return $1;
  }
}

sub increment_version {
  my $version = shift;
  if ($version =~ s/(\d+)$/ $1+1 /e) {
    return $version;
  }
}

sub save_version_in_makefile_pl {
  my $version = shift;
  my $file = `cat Makefile.PL`;
  if (not $file =~ s/VERSION\s*=>\s*['"](\d+[\.\d+]+)['"]/VERSION => '$version'/ ) {
    return undef;
  }
  system( "cp Makefile.PL Makefile.PL.bak" )==0 
    or die "can't save a backup of Makefile.PL";
  if(open M, ">", 'Makefile.PL') {
    print M $file;
    close M;
    return 1;
  }
}

sub get_new_version {
  if (scalar @ARGV) {
    return shift @ARGV;
  }
  my $ver = get_cur_version() || die "can't get current version";
  my $new = increment_version( $ver ) || die "can't increment current version: $ver";
  return $new;
}

sub submit_makefile_pl {
  my $version = shift;
  system( "darcs record -a -m 'release $version' Makefile.PL" )==0 
    or die "can't do 'darcs record' for Makefile.PL";
}


sub make_release {
  my $version = get_new_version() || die 'no new version';
  my $mname   = get_module_name() || die "can't get module name";

  my $assumed_filename = $mname . '-' . $version . '.tar.gz';
  if ( -f $assumed_filename ) {
    die "file $assumed_filename is already present";
  }

  save_version_in_makefile_pl( $version ) or die "can't save new version in Makefile.PL";
  submit_makefile_pl( $version );
  if (open L, ">>", 'release.log') {
    print L "\n";
    print L scalar (localtime), " version: $version\n";
    close L;
  }
  system( 'MAKEMAKEFILE=1 perl Makefile.PL >>release.log' ) ==0 
    or die 'failed at "perl Makefile.PL" step' ;
  system( 'make dist >>release.log' )==0 
    or die 'failed at "make dist" step';

  if ( -f $assumed_filename ) {
    print "file: $assumed_filename\n";
    system( "echo file: $assumed_filename>>release.log" );
  } else {
    die "file $assumed_filename were not made";
  }
}

make_release();

